C      *****************************************************************
C      * SUB DefineProb                                                *
C      * This subroutine does a number of "housekeeping" chores that   *
C      * need to be done before the model is actually run.             *
C      *                                                               *
C      * Variables:                                                    *
C      * A         -Local - Index holder.  Unitless.                   *
C      * ActCorr   -Input - A vector, of size NSpecies, of flags to    *
C      *                    indicate how activity corrections will be  *
C      *                    calculated.                                *
C      *                    1 = None (concentration = activity),       *
C      *                    2 = Davies,                                *
C      *                    3 = Debye-Huckle (not included but easy to *
C      *                        add if desired),                       *
C      *                    4 = Vanselow exchange,                     *
C      *                    5 = Gaines-Thomas exchange.                *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * C         -Local - Loop index.  Unitless.                     *
C      * D         -Local - Loop index.  Unitless.                     *
C      * E         -Local - Index holder.  Unitless.                   *
C      * EqCnst    -Input - A matrix, of size NSpecies by NLayers, of  *
C      *                    the equilibrium constants used for species *
C      *                    formation.  The units depend on            *
C      *                    stochiometric coefficients (SC) and the    *
C      *                    units of species concentrations (SpConc)   *
C      *                    (mol/L).                                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * FirstPhase-Local - A flag for determining whether the first   *
C      *                    phase in the list has been processed.      *
C      *                    Unitless.                                  *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * I         -Local - Loop index.  Unitless.                     *
C      * J         -Local - Loop index.  Unitless.                     *
C      * K         -Local - Loop index.  Unitless.                     *
C      * L         -Local - Loop index.  Unitless.                     *
C      * LinkComp  -Output- A vector, of size NSpecies, that specifies *
C      *                    the component with which a surface-bound   *
C      *                    species is linked.  Unitless.              *
C      *                    (Common block VReac, file VModel.f)        *
C      * LinkList  -Output- A matrix, of size NComp by NSpecies + 1,   *
C      *                    that lists the index numbers of the species*
C      *                    that will be used in the exchange reaction.*
C      *                    The zeroeth position for each row contains *
C      *                    the number of items in that row.  Unitless.*
C      *                    (Common block VReac, file VModel.f)        *
C      * LinkPos   -Output- A vector, of size NComp, of the indices for*
C      *                    the controlling component for a given      *
C      *                    position in the list.  Unitless.           *
C      *                    (Common block VReac, file VModel.f)        *
C      * Lyr       -Local - Layer index.  Unitless.                    *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NCompSize -Input - The max number of components, used to size *
C      *                    arrays.  Unitless.                         *
C      *                    (file ArraySizes.Inc)                      *
C      * NList     -Output- The number of items in the linked list,    *
C      *                    LinkList.  Unitless.                       *
C      *                    (Common block VReac, file VModel.f)        *
C      * NPhases   -Input - The number of phases in the system.        *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * NSpecies  -Input - The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * P         -Local - Index holder.                              *
C      * Q         -Local - Loop index.  Unitless.                     *
C      * S         -Local - Loop index.  Unitless.                     *
C      * tSC       -Output- A matrix, of size NSpecies by NComp, of    *
C      *                    the transformed stoichiometric coefficients*
C      *                    of each comp for each species.  Unitless.  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * tSEC      -Output- A matrix, of size NPhases by NLayers, of   *
C      *                    the transformed solid phase equilibrium    *
C      *                    constants.  The units are consistent with  *
C      *                    units of concentration.                    *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SC        -Output- A matrix, of size NDepSp by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each species.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * SComp     -Output- A vector, of size NPhases, of the          *
C      *                    component associated with each phase.      *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SOrder    -Output- A vector, of size NPhases, of order in     *
C      *                    which to consider phases in mass action    *
C      *                    substitutions.  Unitless.                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SP        -Output- A matrix, of size NPhases by NComps, of    *
C      *                    coefficients.  Unitless.                   *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * Sum       -Local - A variable used for summation.  Units      *
C      *                    consistent with items accumulated.         *
C      * Temp      -Local - Temporary storage.  Units consistent with  *
C      *                    item stored.                               *
C      * tEqnst    -Output- A matrix, of size NSpecies by NLayers, of  *
C      *                    the transformed equilibrium constants.  The*
C      *                    units of tEqCnst  are consistent with the  *
C      *                    units of component concentration.          *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * tSC       -Output- The transformed stoichiometric coeff of    *
C      *                    each comp for each species.                *
C      *                    (common block VSolidPhase, file VModel.f)  *
C      * tSEC      -Output- The transformed solid phase equilibrium    *
C      *                    constants.                                 *
C      *                    (common block VSolidPhase, file VModel.f)  *
C      * tSP       -Output- A transformed SP matrix.                   *
C      *                    (common block VSolidPhase, file VModel.f)  *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      * Z         -In/Out- A vector, of size NComp, of unspeciated    *
C      *                    charge for each component.  Units are      *
C      *                    equivalent charge.                         *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE DEFINEPROB(GOAHEAD)
       IMPLICIT NONE
       INCLUDE 'VMODEL.INC'
       REAL*8 SUM, TEMP

       LOGICAL FIRSTPHASE

       INTEGER A, C, D, E, GOAHEAD, J, K, I, L, LYR, P, Q, S

C      *------------------------------------------------------------*
C      * The first task is to calculate the charge on all species.  *
C      * The charge on components is defined by the user.  The      *
C      * charge on other species (other than components) is         *
C      * calculated from their stoichiometry as defined by the      *
C      * user, thereby assuring internal consistancy.               *
C      *------------------------------------------------------------*
       DO 200 S=NCOMP+1, NSPECIES
          SUM = 0.0
          DO 100 C=1, NCOMP
             SUM = SUM + SC(S, C) * Z(C)
 100      CONTINUE
          Z(S) = SUM
  200  CONTINUE

C      *------------------------------------------------------------*
C      * The next task is to take any solid phases into account in  *
C      * the formulation of the equilibrium problem.  Many of the   *
C      * instructions for doing this are provided by the subroutine *
C      * SelectComp and so that subroutine is called here.          *
C      *------------------------------------------------------------*
       IF (NPHASES.GT.0) CALL SELECTCOMP(GOAHEAD)
       IF (GOAHEAD.EQ.0) GOTO 9999

C      *--------------------------------------------------------*
C      *  Transform saturated phase info by elimination scheme. *
C      *--------------------------------------------------------*
       FIRSTPHASE = .TRUE.
       DO 600 D=1, NPHASES
          E = SORDER(D)
          IF (FIRSTPHASE) THEN
C            *-------------------------------------------*
C            *  Leave first phase in SOrder list as is.  *
C            *-------------------------------------------*
             FIRSTPHASE = .FALSE.
             DO 300 C=1, NCOMP
                TSP(E, C) = SP(E, C)
 300         CONTINUE
             DO 350 LYR=1,NLAYERS
                TSEC(E,LYR) = SEC(E,LYR)
 350         CONTINUE
          END IF

C         *---------------------------------------------*
C         * Transform all the other phases in the list. *
C         *---------------------------------------------*
          DO 500 Q = D + 1, NPHASES
             P = SORDER(Q)
             A = SCOMP(E)
             TEMP = -SP(P, A) / SP(E, A)
             DO 400 C = 1, NCOMP
                IF (C.EQ.A) THEN
                   TSP(P, C) = 0
                ELSE
                   TSP(P, C) = SP(P, C) + SP(E, C) * TEMP
                END IF
 400         CONTINUE
             DO 450 LYR=1,NLAYERS
                TSEC(P,LYR) = SEC(P,LYR) * SEC(E,LYR) ** TEMP
 450         CONTINUE
 500      CONTINUE
 600   CONTINUE

C      *--------------------------------------------*
C      * Copy the information in SC into TSC        *
C      *--------------------------------------------*
       DO 800 S=1, NSPECIES
          DO 700 C=1, NCOMP
             TSC(S, C) = SC(S, C)
 700      CONTINUE
          DO 750 LYR=1, NLAYERS
             TEQCNST(S,LYR) = EQCNST(S,LYR)
 750      CONTINUE
 800   CONTINUE

C      *--------------------------------------------------*
C      * Use the transformed phase info to modify all of  *
C      * the species formation reactions.                 *
C      *--------------------------------------------------*
       IF (NPHASES.GT.0) THEN
C         *-------------------------------------*
C         * First, transform the species coeff. *
C         *-------------------------------------*
          DO 1100 Q=1, NPHASES
             P = SORDER(Q)
             A = SCOMP(P)
             DO 1000 S=NCOMP+1, NSPECIES
                DO 900 C=1, NCOMP
                   TSC(S, C) = TSC(S, C) - ((TSC(S, A) * TSP(P, C)) / 
     >                         TSP(P, A))
C                  *----------------------------------------------*
C                  * The following IF block is an attempt to trap *
C                  * possible roundoff errors.  It is unlikely    *
C                  * that the above formula should produce small  *
C                  * values of coefficients so they are caught    *
C                  * here and set to zero. It might be wise to    *
C                  * write a message to a log file or to the      *
C                  * screen when this happens, just to be sure    *
C                  * that it wasn't supposed to happen.           *
C                  *---------------------------------------------
                   IF ((ABS(TSC(S, C)).LT.0.1).AND.
     >                (ABS(TSC(S, C)).NE.0)) THEN
                      WRITE(YASOUT,*) 
     >               'ROUNDOFF ERROR IN DEFINEPROB'
                      TSC(S, C) = 0
                   END IF
  900           CONTINUE
               DO 950 LYR=1,NLAYERS
                  TEQCNST(S,LYR) = TEQCNST(S,LYR) * 
     >                        (TSEC(P,LYR) ** (SC(S, A) / TSP(P, A)))
  950          CONTINUE
 1000        CONTINUE
 1100     CONTINUE

C         *------------------------------*
C         * Next, the component coeff.   *
C         *------------------------------*
          DO 1300 J=1, NCOMP
             IF ((STYPE(J).EQ.SOLGAS).OR.(STYPE(J).EQ.SPSOLGAS)) THEN
                TSC(J, J) = 0
                P = SPHASE(J)
                DO 1200 K=1, NCOMP
                   IF (K.NE.J) THEN
                      TSC(J, K) = - TSP(P, K) 
                   END IF
 1200           CONTINUE
                DO 1250 LYR=1,NLAYERS
                  TEQCNST(J,LYR) = TSEC(P,LYR) ** (1 / TSP(P, J))
 1250           CONTINUE
             ELSE
                TSC(J, J) = SC(J, J)
             END IF
 1300     CONTINUE
       END IF

C      *------------------------------------------------*
C      *  Create list of linked species and components. *
C      *------------------------------------------------*

C      *----------------------------------------------------*
C      * First, figure out how many lists there need to be. *
C      *----------------------------------------------------*
       I = 0
       DO 1400 C=1, NCOMP
          IF ((ACTCORR(C).EQ.4).OR.(ACTCORR(C).EQ.5)) THEN
             I = I + 1
             LINKPOS(C) = I
             LINKLIST(I, 1) = C
             LINKCOMP(C) = C
          END IF
 1400  CONTINUE
       NLIST = I

C      *------------------------*
C      * Now, create each list. *
C      *------------------------*
       DO 1600 L = 1, NLIST
          C = LINKLIST(L,1)
          P = 1
          DO 1500 S = NCOMP + 1, NSPECIES
             IF (SC(S, C).NE.0) THEN
               IF (ACTCORR(S).EQ.ACTCORR(C)) THEN
                   P = P + 1
                   LINKLIST(L, P) = S
                   LINKCOMP(S) = C
                ELSE
C                  *-------------------------------------------*
C                  * This means trouble, incompatible exchange *
C                  * formulations were specified for this      *
C                  * combination of species and component.     *
C                  * Create error message and terminate        *
C                  * program.                                  *
C                  *-------------------------------------------*
                   GOAHEAD = 0
                   WRITE(6,*) 'Error in routine DefineProb'
                   WRITE(6,*) 'Incompatible exchange formulations'
                   WRITE(YASOUT,*) 'Error in routine DefineProb'
                   WRITE(YASOUT,*) 'Incompatible exchange formulations'
                   GOTO 9999
                END IF
             END IF
 1500     CONTINUE
          LINKLIST(L, 0) = P
 1600  CONTINUE

C      *------------------------------------------------------*
C      * Check that proper activity correction is specified   *
C      * for each species according to type (i.e., 1, 2, or   *
C      * 3 for aqueous, and 1, 4, or 5 for surface species).  *
C      *------------------------------------------------------*
       DO 1700 S = 1, NSPECIES
          P = ACTCORR(S)
          IF (STYPE(S).LE.SOLGAS) THEN
C            *---------------------------------------*
C            * Aqueous species,                      *
C            * only ACTCORR = 1, 2, or 3 is allowed. *
C            *---------------------------------------*
             IF (.NOT.((P.EQ.1).OR.(P.EQ.2).OR.(P.EQ.3))) THEN 
C               *---------------------------------------*
C               * Print an error and terminate program. *
C               *---------------------------------------*
                GOAHEAD = 0
                WRITE(6,*) 'Error in routine DefineProb'
                WRITE(6,*) 'Aqueous has wrong ActCorr'
                WRITE(YASOUT,*) 'Error in routine DefineProb'
                WRITE(YASOUT,*) 'Aqueous has wrong ActCorr'
                GOTO 9999
             END IF
          ELSE IF (STYPE(S).GE.SPMASBAL) THEN
C            *---------------------------------------*
C            * Surface species                       *
C            * only ACTCORR = 1, 4, or 5 is allowed. *
C            *---------------------------------------*
             IF (.NOT.((P.EQ.1).OR.(P.EQ.4).OR.(P.EQ.5))) THEN
C               *---------------------------------------*
C               * Print an error and terminate program. *
C               *---------------------------------------*
                GOAHEAD = 0
                WRITE(6,*) 'Error in routine DefineProb'
                WRITE(6,*) 'Surface has wrong ActCorr'
                WRITE(YASOUT,*) 'Error in routine DefineProb'
                WRITE(YASOUT,*) 'Surface has wrong ActCorr'
                GOTO 9999
             END IF
          ELSE
C               *---------------------------------------*
C               * Improper species type was specified.  *
C               * Print an error and terminate program. *
C               *---------------------------------------*
                GOAHEAD = 0
                WRITE(6,*) 'Error in routine DefineProb'
                WRITE(6,*) 'Improper species type'
                WRITE(YASOUT,*) 'Error in routine DefineProb'
                WRITE(YASOUT,*) 'Improper species type'
                GOTO 9999
          END IF
 1700   CONTINUE

C      *--------------*
C      * Escape hatch *
C      *--------------*
 9999  CONTINUE

       RETURN
       END
C      ****************************************************************
C      *               END OF SUBROUTINE DEFPROB                      *
C      ****************************************************************

